﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013

*/
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace Inet.Viewer.WinForms.Export
{
    /// <summary>
    /// Panel for selection an output format.
    /// </summary>
    public partial class FormatSelectControl : UserControl
    {
        private FormatSettingsControl selectedFormatSettingsControl;
        private FormatSettingsControl[] settingsControls;
        private HashSet<string> allowedFormats;

        /// <summary>
        /// Occured when the format is changed.
        /// </summary>
        public event EventHandler FormatChanged;

        /// <summary>
        /// Creates the panel wth the specified formats.
        /// </summary>
        /// <param name="settingControls">the format settings controls</param>
        public FormatSelectControl(FormatSettingsControl[] settingControls)
        {
            this.settingsControls = settingControls;
            InitializeComponent();
            UpdateButtons();
        }

        /// <summary>
        /// Returns the currently selected format control.
        /// </summary>
        public FormatSettingsControl SelectedFormatSettingsControl
        {
            get
            {
                return selectedFormatSettingsControl;
            }
        }

        /// <summary>
        /// Called when the user clicks a format button. 
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void OnButtonCheckedChanged(object sender, EventArgs e)
        {
            RadioButton checkBox = (RadioButton)sender;
            selectedFormatSettingsControl = (FormatSettingsControl)checkBox.Tag;
            if (FormatChanged != null)
            {
                FormatChanged.Invoke(this, new EventArgs());
            }
        }

        /// <summary>
        /// Sets the set of allowed formats.
        /// </summary>
        public HashSet<string> AllowedFormats
        {
            set
            {
                allowedFormats = value;
                foreach (FormatSettingsControl settingsControl in settingsControls)
                {
                    settingsControl.AllowedFormats = value;
                }
                UpdateButtons();
            } 
        }

        /// <summary>
        /// Create all buttons with allowed formats.
        /// </summary>
        private void UpdateButtons()
        {
            flowLayoutPanel1.Controls.Clear();
            Font labelFont = new System.Drawing.Font("Microsoft Sans Serif", 8F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, (byte)0);
            bool first = true;
            selectedFormatSettingsControl = null;
            foreach (FormatSettingsControl settingsControl in settingsControls)
            {
                if (!settingsControl.Allowed)
                {
                    continue;
                }
                RadioButton button = new RadioButton();
                button.Appearance = System.Windows.Forms.Appearance.Button;
                button.AutoSize = true;
                button.Font = labelFont;
                button.Location = new System.Drawing.Point(3, 3);
                button.MinimumSize = new System.Drawing.Size(80, 80);
                button.Size = new System.Drawing.Size(80, 80);
                button.TextAlign = System.Drawing.ContentAlignment.BottomCenter;
                button.UseVisualStyleBackColor = true;
                button.Image = settingsControl.Icon;
                button.Text = settingsControl.Label;
                button.Tag = settingsControl;
                button.TabIndex = 10;
                button.Click += new System.EventHandler(this.OnButtonCheckedChanged);
                flowLayoutPanel1.Controls.Add(button);
                if (first)
                {
                    selectedFormatSettingsControl = settingsControl;
                    button.Checked = true;
                    first = false;
                }
            }
        }

        /// <summary>
        /// Returns a flag indicating whether any format is available to export.
        /// </summary>
        public bool AnyFormatAvailable { get { return flowLayoutPanel1.Controls.Count > 0; } }
    }
}
